<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Transaction;
use App\Notifications\InvestmentCompletedNotification;


class UserVipInvestment extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id', 
        'vip_product_id',
        'amount',
        'daily_earnings',
        'last_payout_at',
        'expires_at',
        'type',
        'is_active',
          'amount_returned',
    ];
    
     protected $casts = [
        'last_payout_at' => 'datetime',
        'expires_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

  // app/Models/UserVipInvestment.php
public function isDueForPayout()
{
    if (!$this->last_payout_at) {
        return true;
    }
    
    return now()->diffInMinutes($this->last_payout_at) >= 1; // 1 minute check

 
}
// app/Models/UserVipInvestment.php

public function scopeDueForPayout($query)
{
    return $query->where(function($q) {
        $q->whereNull('last_payout_at')
          ->orWhere('last_payout_at', '<=', now()->subHours(24));
    });
}




 // app/Models/UserVipInvestment.php

// UserVipInvestment মডেলে এই পরিবর্তনগুলো করুন

public static function processExpiredInvestments()
{
    $expiredInvestments = self::where('expires_at', '<=', now())
                            ->where('is_active', true)
                            ->where('amount_returned', false)
                            ->with(['user', 'product'])
                            ->get();
    
    foreach ($expiredInvestments as $investment) {
        DB::transaction(function () use ($investment) {
            // ১. ব্যালেন্সে টাকা ফেরত যোগ করুন
            $user = $investment->user;
            $user->balance += $investment->amount;
            $user->save();
            
            // ২. ট্রানজেকশন হিস্ট্রি রেকর্ড করুন
            Transaction::create([
                'user_id' => $user->id,
                'amount' => $investment->amount,
                'type' => 'vip_investment_return',
                'status' => 'completed',
                'notes' => $investment->product->investment_tool.' থেকে বিনিয়োগ ফেরত'
            ]);
            
            // ৩. ইনভেস্টমেন্ট স্ট্যাটাস আপডেট করুন
            $investment->update([
                'is_active' => false,
                'amount_returned' => true
            ]);
            
            // ৪. নোটিফিকেশন পাঠান (ঐচ্ছিক)
            $user->notify(new InvestmentCompletedNotification($investment));
        });
    }
    
    return $expiredInvestments->count();
}
    // বাংলাদেশের সময়ে ফরম্যাট করার মেথড
    public function getBangladeshTime($date)
    {
        return Carbon::parse($date)->timezone('Asia/Dhaka');
    }
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function product()
    {
        return $this->belongsTo(VipProduct::class, 'vip_product_id');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where('expires_at', '>', now());
    }

    public function totalEarnings()
    {
        if (!$this->last_payout_at) {
            return $this->daily_earnings;
        }
        
        $daysEarned = $this->created_at->diffInDays($this->last_payout_at);
        return $daysEarned * $this->daily_earnings;
    }

    public function nextPayoutTime()
    {
        if (!$this->last_payout_at) {
            return now();
        }
        
        return $this->last_payout_at->addDay();
    }

       // বাংলাদেশের তারিখ ফরম্যাটCommands
    public function formattedExpiresAt()
    {
        return $this->expires_at 
            ? $this->getBangladeshTime($this->expires_at)->format('d M Y')
            : 'তারিখ পাওয়া যায়নি';
    }
}