<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use App\Models\Withdrawal;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class WithdrawalController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        // wallet_info কে অ্যারে হিসেবে নিশ্চিত করুন
        $walletInfo = is_array($user->wallet_info) ? $user->wallet_info : [];
        
        $methods = PaymentMethod::where('status', true)
                    ->where('is_active', true)
                    ->orderBy('order')
                    ->get();
        
        $withdrawals = Withdrawal::where('user_id', $user->id)
                        ->with('paymentMethod')
                        ->latest()
                        ->paginate(10);
                    
        return view('user.transactions.withdraw', [
            'methods' => $methods,
            'withdrawals' => $withdrawals,
            'wallet_info' => $walletInfo
        ]);
    }

  public function store(Request $request)
{
    \Log::info('Withdrawal request received:', $request->all());

    $user = Auth::user();
    
    $request->validate([
        'amount' => 'required|numeric|min:500|max:999999',
        'payment_method_id' => 'required|exists:payment_methods,id',
        'wallet_info' => 'required|array',
        'wallet_info.*' => 'nullable|string|regex:/^01[3-9]\d{8}$/'
    ], [
        'amount.min' => 'ন্যূনতম উত্তোলন পরিমাণ ৫০০ ৳',
        'amount.max' => 'সর্বোচ্চ উত্তোলন পরিমাণ ৯,৯৯,৯৯৯ ৳',
        'wallet_info.*.regex' => 'বৈধ বাংলাদেশী মোবাইল নম্বর লিখুন (01XXXXXXXXX)'
    ]);

    $method = PaymentMethod::findOrFail($request->payment_method_id);

    // ১৫% ফি ক্যালকুলেশন
    $feePercentage = 15; // 15%
    $feeAmount = ($request->amount * $feePercentage) / 100;
    $netAmount = $request->amount - $feeAmount;

    // ব্যালেন্স চেক (পুরো উত্তোলন পরিমাণ + ফি)
    if ($user->balance < $request->amount) {
        return back()->with('error', 'আপনার ব্যালেন্স অপর্যাপ্ত!')->withInput();
    }

    // উত্তোলন লিমিট চেক
    if ($request->amount < $method->min_withdraw) {
        return back()->with('error', "সর্বনিম্ন উত্তোলন পরিমাণ {$method->min_withdraw} ৳")->withInput();
    }

    if ($method->max_withdraw && $request->amount > $method->max_withdraw) {
        return back()->with('error', "সর্বোচ্চ উত্তোলন পরিমাণ {$method->max_withdraw} ৳")->withInput();
    }

    // ওয়ালেট নাম্বার হ্যান্ডলিং
    $walletInfo = $user->wallet_info ?? [];
    if (isset($walletInfo[$method->name])) {
        $walletNumber = $walletInfo[$method->name];
    } else {
        if (empty($request->wallet_info[$method->name])) {
            return back()->with('error', 'অনুগ্রহ করে আপনার ওয়ালেট নম্বর প্রদান করুন')->withInput();
        }

        $walletNumber = $request->wallet_info[$method->name];
        $walletInfo[$method->name] = $walletNumber;
        $user->wallet_info = $walletInfo;
        
        if (!$user->save()) {
            return back()->with('error', 'ওয়ালেট তথ্য সংরক্ষণ করতে ব্যর্থ হয়েছে')->withInput();
        }
    }

    DB::beginTransaction();

    try {
        // ব্যালেন্স আপডেট (পুরো উত্তোলন পরিমাণ কাটা হবে)
        $user->balance -= $request->amount;
        $user->save();

        // উত্তোলন রেকর্ড তৈরি
        $withdrawal = Withdrawal::create([
            'user_id' => $user->id,
            'payment_method_id' => $method->id,
            'sender_number' => $walletNumber,
            'amount' => $netAmount, // নেট পরিমাণ সেভ করা হচ্ছে (ফি বাদে)
            'fee' => $feeAmount, // ফি সেভ করা হচ্ছে
            'gross_amount' => $request->amount, // মূল উত্তোলন পরিমাণ
            'status' => 'pending',
            'transaction_id' => 'WDR' . time() . $user->id
        ]);

        // ফি হিসাবে একটি ট্রানজেকশন রেকর্ড তৈরি করুন
        \App\Models\Transaction::create([
            'user_id' => $user->id,
            'type' => 'withdrawal_fee',
            'amount' => $feeAmount,
            'method' => 'system',
            'status' => 'approved',
            'transaction_id' => 'FEE-'.uniqid(),
            'notes' => 'Withdrawal fee (15%) for withdrawal #'.$withdrawal->id
        ]);

        DB::commit();

        return redirect()->route('withdraw.index')
            ->with('success', "উত্তোলনের অনুরোধ সফলভাবে জমা হয়েছে! নেট পরিমাণ: {$netAmount} ৳ (১৫% ফি কাটা হয়েছে)");

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->with('error', 'একটি ত্রুটি ঘটেছে: '.$e->getMessage())->withInput();
    }
}
}